'use client'
import React, { useState, useEffect, useRef } from 'react';
import styles from "../../assets/style.module.css";
import { useSession } from "next-auth/react";
import Loading from '../loading';
import Swal from 'sweetalert2'
import { signOut } from 'next-auth/react';

const ClientComponent = () => {
    const [investorPoftfolio, setInvestorPortfolio] = useState([])
    const [isLoading, setIsLoading] = useState(false)
    const { data } = useSession();
    const prevToken = useRef(null);

    useEffect(() => {
        let token;
        if (data && data.user && data.user.token && data.user.token !== prevToken.current) {
            token = data.user.token;
            prevToken.current = token
            setIsLoading(true);
            fetchPortfolioData(token);
        }

    }, [data?.user?.token]);

    function formatIndianNumber(value) {
        if (value === null || value === undefined || isNaN(value)) {
            return "0";
        }
    
        const isNegative = value < 0; // Check if the number is negative
        const absoluteValue = Math.abs(value).toString(); // Get the absolute value
    
        const parts = absoluteValue.split("."); // Split into whole and fractional parts
        const wholePart = parts[0];
        const fractionalPart = parts[1] ? `.${parts[1]}` : "";
    
        // Correct regex for Indian numbering system
        const formattedWholePart = wholePart.replace(
            /(\d+)(\d{3})$/,
            (match, p1, p2) => p1.replace(/\B(?=(\d{2})+(?!\d))/g, ",") + "," + p2
        );
    
        const formattedNumber = formattedWholePart + fractionalPart;
    
        return isNegative ? `(${formattedNumber})` : formattedNumber; // Add parentheses for negative numbers
    }
    
    

    const fetchPortfolioData = async (token) => {
        try {
            const url = process.env.NEXT_PUBLIC_API_URL + '/dashboard-summary-info';
            const res = await fetch(url, {
                method: "GET",
                headers: {
                    "Accept": "application/json",
                    'Content-Type': 'application/json',
                    'Authorization': 'Bearer ' + String(token)
                }
            });
            const data = await res.json();
            if (data.status == 'success') {
                console.log(data.dashboard_summary);
                setInvestorPortfolio(data.dashboard_summary[0])
            } else if (res.status == 401) {
                Swal.fire({
                    position: "top-end",
                    icon: "warning",
                    title: data.message,
                    showConfirmButton: false,
                    timer: 1000,
                }).then((result) => {
                    if (result.dismiss === Swal.DismissReason.timer) {
                        signOut();
                    }
                });
            }
        } catch (error) {
            if (error.name == 'TypeError') {
                Swal.fire({
                    position: "top-end",
                    icon: "error",
                    title: 'The server is not responding. Please wait for a moment.',
                    showConfirmButton: false,
                    timer: 2000,
                });
            }
        } finally {
            setIsLoading(false)
        }
    };

    if (isLoading) {
        return (
            <Loading loading={isLoading} />
        )
    }

    return (
        <>
            <div className={`${styles.d_flex} ${styles.w_100} ${styles.mx_20} ${styles.mt_sm_35}`}>
                {investorPoftfolio && Object.keys(investorPoftfolio).length > 0 && (
                    <div className={` ${styles.w_100} ${styles.mt_sm_35}`}>
                        <div className={`${styles.d_flex} ${styles.w_100} ${styles.mt_sm_35}`}>
                            <h1 className={`${styles.gradient_text_green} ${styles.w_100} ${styles.text_center}`}>Welcome to the Dashboard,  {investorPoftfolio.investor_name}</h1>

                        </div>
                        <div className={`${styles.d_flex} ${styles.w_100} ${styles.mt_sm_35}`}>
                            <h1 className={`${styles.gradient_text_green} ${styles.w_100} ${styles.text_center}`}>As on Date,  {investorPoftfolio.as_on_date}</h1>

                        </div>
                        <div className={`${styles.d_flex} ${styles.w_100} ${styles.mt_sm_35} ${styles.my_20}`}> 
                            <section className={`${styles.info_boxes}  ${styles.w_100} `}> 
                            <div className={`${styles.box}`}>
                                Cost Value: {formatIndianNumber(parseFloat(investorPoftfolio.cost_value_securities || 0))}
                            </div>
                            <div className={`${styles.box}`}>
                                Market Value: {formatIndianNumber(parseFloat(investorPoftfolio.market_value_securities || 0))}
                            </div>
                            <div className={`${styles.box}`}>
                                Equity at Market Value: {formatIndianNumber(parseFloat(investorPoftfolio.equity_at_market || 0))}
                            </div>
                            <div className={`${styles.box}`}>
                                CRM Ratio: {investorPoftfolio.cmr_ration || "N/A"}
                            </div>
                            <div className={`${styles.box}`}>
                                Purchase Power: {formatIndianNumber(parseFloat(investorPoftfolio.purchase_power || 0))}
                            </div>
                            <div className={`${styles.box}`}>
                                Net Gain Loss: {formatIndianNumber(parseFloat(investorPoftfolio.net_gain_loss || 0))}
                            </div> 
                            </section>
                        </div>  
                    </div>   
                )}
            </div>
        </>
    );
};

export default ClientComponent;
